/*****************************************************************************
 * ps.c: Program Stream demux module for VLC.
 *****************************************************************************
 * Copyright (C) 2004 the VideoLAN team
 * $Id: ps.c 14719 2006-03-11 17:35:55Z zorglub $
 *
 * Authors: Laurent Aimar <fenrir@via.ecp.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston MA 02110-1301, USA.
 *****************************************************************************/

/*****************************************************************************
 * Preamble
 *****************************************************************************/
#include <stdlib.h>                                      /* malloc(), free() */

#include <vlc/vlc.h>
#include <vlc/input.h>

#include "ps.h"

/* TODO:
 *  - re-add pre-scanning.
 *  - ...
 */

/*****************************************************************************
 * Module descriptor
 *****************************************************************************/
static int  Open   ( vlc_object_t * );
static int  OpenAlt( vlc_object_t * );
static void Close  ( vlc_object_t * );

vlc_module_begin();
    set_description( _("MPEG-PS demuxer") );
    set_category( CAT_INPUT );
    set_subcategory( SUBCAT_INPUT_DEMUX );
    set_capability( "demux2", 1 );
    set_callbacks( Open, Close );
    add_shortcut( "ps" );

    add_submodule();
    set_description( _("MPEG-PS demuxer") );
    set_capability( "demux2", 9 );
    set_callbacks( OpenAlt, Close );
vlc_module_end();

/*****************************************************************************
 * Local prototypes
 *****************************************************************************/

struct demux_sys_t
{
    ps_psm_t    psm;
    ps_track_t  tk[PS_TK_COUNT];

    int64_t     i_scr;
    int         i_mux_rate;

    vlc_bool_t  b_lost_sync;
    vlc_bool_t  b_have_pack;
};

static int Demux  ( demux_t *p_demux );
static int Control( demux_t *p_demux, int i_query, va_list args );

static int      ps_pkt_resynch( stream_t *, uint32_t *pi_code );
static block_t *ps_pkt_read   ( stream_t *, uint32_t i_code );

/* moyekj START */
/*
 * Author: Kevin J. Moye - 8/04/2004
 * Code for identifying commercial start/stop points (time & file positions)
 * from ReplayTV 4K & 5K series .ndx and .evt files.
 * The intent is to integrate this code with VideoLAN VLC to enable
 * Automatic Commercial Advance while playing or streaming an RTV mpeg.
 *
 * This code is HEAVILY based on replaypc and rtvtools code for
 * parsing .ndx and .evt files.
 */
#include <assert.h>
#include <memory.h>
#include <sys/stat.h>
//#include <osd.h>
#include <vlc/vout.h>
#include "network.h"

#include "rtvparse.h"

int RTV_DEBUG = 0;

// From rtv.c (replaypc)
u8 rtv_to_u8(unsigned char ** pp) 
{
    unsigned char * p;
    u8 r;

    p = *pp;
    r = *p++;
    *pp = p;

    return r;
}

u16 rtv_to_u16(unsigned char ** pp)
{
    u16 r;
    r  = rtv_to_u8(pp) << 8;
    r |= rtv_to_u8(pp);
    return r;
}

u32 rtv_to_u32(unsigned char ** pp)
{
    u32 r;
    r  = rtv_to_u16(pp) << 16;
    r |= rtv_to_u16(pp);
    return r;
}

u64 rtv_to_u64(unsigned char ** pp)
{
    u64 r;

    r  = (u64)(rtv_to_u32(pp)) << 32;
    r |= (u64)(rtv_to_u32(pp));
    return r;
}

void rtv_from_u8(unsigned char ** pp, u8 v)
{
    unsigned char * p;

    p = *pp;
    *p++ = v;
    *pp = p;
}
        
void rtv_from_u16(unsigned char ** pp, u16 v)
{
    rtv_from_u8(pp, (u8)((v >> 8) & 0x00ff));
    rtv_from_u8(pp, (u8)((v     ) & 0x00ff));
}

void rtv_from_u32(unsigned char ** pp, u32 v)
{
    rtv_from_u16(pp, (u16)((v >> 16) & 0x0000ffff));
    rtv_from_u16(pp, (u16)((v      ) & 0x0000ffff));
}

void rtv_from_u64(unsigned char ** pp, u64 v)
{
    rtv_from_u32(pp, (u32)((v >> 32) & 0x00000000ffffffffLLU));
    rtv_from_u32(pp, (u32)((v      ) & 0x00000000ffffffffLLU));
}


// From ndx.c (replaypc)
// 4K series ndx single record parse
static char * format_seconds(float seconds)
{
    static char buffer[12];
    int minutes;

    minutes = (int)seconds / 60;
    seconds -= minutes * 60;

    if (minutes)
        sprintf(buffer, "%3d:", minutes);
    else
        strcpy(buffer, "    ");
    sprintf(buffer + 4, "%07.4f", seconds);

    return buffer;
}

int read_ndx4k_record(unsigned char **ndxbuf, struct ndx4k_record * pr)
{
    memset(pr, 0, 32);
        
    pr->flag_1            = *(*ndxbuf)++;
    pr->commercial_flag   = *(*ndxbuf)++;
    pr->video_offset      = rtv_to_u16(ndxbuf);
    pr->unk_fe            = *(*ndxbuf)++;
    pr->macrovision       = *(*ndxbuf)++;
    pr->macrovision_count = rtv_to_u16(ndxbuf);
    pr->audio_offset      = rtv_to_u32(ndxbuf);
    pr->unused1           = rtv_to_u32(ndxbuf);
    pr->timestamp         = rtv_to_u64(ndxbuf);
    pr->stream_position   = rtv_to_u64(ndxbuf);

    return 32;
}

int rtvError(char *message)
{
   fprintf(stderr,"%s\n",message);
   return(0);
}

static void unexpected4k(int recno, const char * reason, u64 value)
{
 char msg[256];
   sprintf(msg, "UNEXPECTED %d: %s; value == %llu\n",
      recno, reason, value
   );
   rtvError(msg);
}

// 4K series ndx header check
int check_header4k(struct ndx4k_record * pr)
{
    if (pr->flag_1 != 0x02) {
        unexpected4k(-1, "Header flag_1 not 0x02", pr->flag_1);
        return 0;
    }
    if (pr->commercial_flag != 0x02) {
        unexpected4k(-1, "Header commercial_flag not 0x02",
           pr->commercial_flag
        );
        return 0;
    }
    if (pr->video_offset != 0) {
        unexpected4k(-1, "Header video_offset not 0", pr->video_offset);
        return 0;
    }
    if (pr->unk_fe != 0) {
        unexpected4k(-1, "Header unk_fe not 0", pr->unk_fe);
        return 0;
    }
    if (pr->macrovision != 0) {
        unexpected4k(-1, "Header macrovision not 0", pr->macrovision_count);
        return 0;
    }
    if (pr->macrovision_count != 0) {
        unexpected4k(-1, "Header macrovision_count not 0",
           pr->macrovision_count
        );
        return 0;
    }
    if (pr->audio_offset != 0) {
        unexpected4k(-1, "Header audio_offset not 0", pr->audio_offset);
        return 0;
    }
    if (pr->unused1 != 0) {
        unexpected4k(-1, "Header unused1 not 0", pr->unused1);
        return 0;
    }
    if (pr->timestamp != 0) {
        unexpected4k(-1, "Header timestamp not 0", pr->timestamp);
        return 0;
    }
    if (pr->stream_position != 0) {
        unexpected4k(-1, "Header stream_position not 0", pr->stream_position);
        return 0;
    }
    return 1;
}

// 4K series ndx file parser to determine commercial segments
static int rtvParse4kNdx(
   unsigned char *ndxbuf, long int ndxSize,
   struct rtvCommercialMarks ***s, int *count,
   u64 minCommercial
)
{
 unsigned char buf[1024], *p;
 struct 
 {
   struct ndx4k_record r;
   float             seconds;
   int               bad_video;
   int               bad_audio;
 } this, last;
 int recno    = 0;
 long int size=0;
 u64 basetime = 0;
 double commercial_start_seconds = 0;
 u64    commercial_start_pos = 0;
 int in_commercial = 0;
 char t1[16], t2[16], t3[16];

   memset(&this, 0, sizeof this);
   size = read_ndx4k_record(&ndxbuf, &this.r);
   if( !check_header4k(&this.r) ) return 0;

   if(RTV_DEBUG) {
      fprintf(stderr,"87654321 SC Recno VidO  AudO File Pos   Time Offs VideoPos AudioPos VAM\n");
   }

   last = this;
   memset(&this, 0, sizeof this);
   while ( (size += read_ndx4k_record(&ndxbuf, &this.r)) <= ndxSize ) {
      /* 0x8000 -- unknown
         0x4000 -- unknown, new with jan 30 sw
         0x2000 -- PPV/protected (?? not any more?)
         0x0800 -- unknown, new with jan 30 sw
         0x0400 -- unknown, new with jan 30 sw
         0x0200 -- unknown, new with jan 30 sw
         0x0100 -- unknown, new with jan 30 sw
         0x0002 -- black screen
         0x0001 -- commercial
      */
      if (this.r.commercial_flag & ~0x03) {
         unexpected4k(recno, "commercial flag has unexpected bit set",
            this.r.commercial_flag
         );
         return 0;
      }
            
      //if (this.r.unk_fe != 0xfe && this.r.macrovision == 0) {
      //   unexpected4k(recno, "unk_fe != 0xfe without macrovision set",
      //      this.r.unk_fe
      //   );
      //}
      if (this.r.macrovision != 0 && this.r.macrovision != 3) {
         unexpected4k(recno, "macrovision != 0 or 3", this.r.macrovision);
         return 0;
      }
      //if (this.r.macrovision == 0 && this.r.macrovision_count != 0) {
      //   unexpected4k(recno, "macrovision clear but count != 0",
      //      this.r.macrovision_count
      //   );
      //}
      if (this.r.macrovision && this.r.macrovision_count == 0) {
         unexpected4k(recno, "macrovision count clear but macrovision set",
            this.r.macrovision
         );
         return 0;
      }
      if (this.r.unused1 != 0) {
         unexpected4k(recno, "unused1 != 0", this.r.unused1);
         return 0;
      }
      if (this.r.stream_position % 0x8000 != 0) {
         unexpected4k(recno, "stream position not 32k boundary",
            this.r.stream_position
         );
         return 0;
      }
      if (this.r.video_offset % 4 != 0) {
         unexpected4k(recno, "video_offset % 4 != 0", this.r.video_offset);
         return 0;
      }
      if (this.r.video_offset >= 0x8000) {
         unexpected4k(recno, "video_offset larger than cluster size",
            this.r.video_offset
         );
         return 0;
      }
      if (this.r.audio_offset % 4 != 0) {
         unexpected4k(recno, "audio_offset % 4 != 0", this.r.audio_offset);
         return 0;
      }
            
      if (recno == 0) {
         basetime   = this.r.timestamp;
      }
      this.seconds = ((s64)(this.r.timestamp - basetime)) / 1000000000.0;

      p = buf;
      if (this.r.commercial_flag & 0x1) {
         if (!in_commercial) {
            commercial_start_seconds = this.r.timestamp - basetime;
            commercial_start_pos     = this.r.stream_position;
            in_commercial = 1;
            if (!(this.r.commercial_flag & 0x02)) {
               unexpected4k(recno, "Start of commercial without 0x02 flag",
                  this.r.commercial_flag
               );
               in_commercial = 0;
            }
         }
      } else if (in_commercial) {
       if((this.r.timestamp - basetime - commercial_start_seconds) > minCommercial) {
         if(RTV_DEBUG) {
            fprintf(stderr,"# Commercial %s - ",
               format_ts(commercial_start_seconds, t1)
            );
            fprintf(stderr,"%s ", format_ts(this.r.timestamp - basetime, t2));
            fprintf(stderr,"(%s)\n",
               format_ts(
                  this.r.timestamp - basetime - commercial_start_seconds, t3
               )
            );
         }
         in_commercial = 0;

         if( (*count) == 0 ) {
           (*s) = (struct rtvCommercialMarks **)malloc(
              sizeof(struct rtvCommercialMarks *)
           );
         } else {
           (*s) = (struct rtvCommercialMarks **)realloc(
              (*s), sizeof(struct rtvCommercialMarks *)*((*count)+1)
           );
         }
         (*s)[(*count)] = (struct rtvCommercialMarks *)malloc(
            sizeof(struct rtvCommercialMarks)
         );
         (*s)[(*count)]->tstart = commercial_start_seconds;
         (*s)[(*count)]->fstart = commercial_start_pos;
         (*s)[(*count)]->tstop  = this.r.timestamp - basetime;
         (*s)[(*count)]->fstop  = this.r.stream_position;
         (*count)++;
       }
      }
            
      if(RTV_DEBUG) {
         fprintf(stderr, "%c", this.r.flag_1 & 0x80 ? 'X' : '.');
         fprintf(stderr, "%c", this.r.flag_1 & 0x40 ? 'X' : '.');
         fprintf(stderr, "%c", this.r.flag_1 & 0x20 ? 'X' : '.');
         fprintf(stderr, "%c", this.r.flag_1 & 0x10 ? 'X' : '.');
         fprintf(stderr, "%c", this.r.flag_1 & 0x08 ? 'X' : '.');
         fprintf(stderr, "%c", this.r.flag_1 & 0x04 ? 'X' : '.');
         fprintf(stderr, "%c", this.r.flag_1 & 0x02 ? 'X' : '.');
         fprintf(stderr, "%c ", this.r.flag_1 & 0x01 ? 'X' : '.');
         fprintf(stderr, "%c", this.r.commercial_flag & 0x02 ? 'X' : '.');
         fprintf(stderr, "%c ", this.r.commercial_flag & 0x01 ? 'X' : '.');
         fprintf(stderr, "%5d ", recno);
         fprintf(stderr, "%4x ", this.r.video_offset);
         fprintf(stderr, "%5lx ", this.r.audio_offset);
         fprintf(stderr, "%08lld ", this.r.stream_position);
         fprintf(stderr, "%s ", format_seconds(this.seconds));
         fprintf(stderr, "%08Lx ",
            this.r.stream_position + this.r.video_offset
         );
         fprintf(stderr, "%08Lx ",
            this.r.stream_position + this.r.video_offset + this.r.audio_offset
         );
         fprintf(stderr, "%c",
            this.bad_video ? last.bad_video ? 'O' : 'o' : '.'
         );
         fprintf(stderr, "%c",
            this.bad_audio ? last.bad_audio ? 'O' : 'o' : '.'
         );
         fprintf(stderr, "%c\n", this.r.macrovision ? 'O' : '.');
      }

      recno++;
      last = this;
      memset(&this, 0, sizeof this);
   }
   return 1;
}

// 5K series ndx header parse (32 bytes)
int read_ndx5k_header(unsigned char **ndxbuf)
{
   *ndxbuf += 32;
   return 32;
}

// 5K series ndx single record parse
int read_ndx5k_record(unsigned char **ndxbuf, struct ndx5k_record * pr)
{
    memset(pr, 0, 24);
        
    pr->timestamp      = rtv_to_u64(ndxbuf);
    pr->filepos_iframe = rtv_to_u64(ndxbuf);
    pr->iframe_size    = rtv_to_u32(ndxbuf);
    pr->empty          = rtv_to_u32(ndxbuf);

    return 24;
}

// From evtdump.c (rtvtools)
char* format_ts(INT64 ts, char *time)
{
 UINT32 ms = (UINT32)(ts / 1000000);
 UINT32 sec = ms / 1000;
 UINT32 min = sec / 60;

   sec -= (min*60);
   ms %= 1000;
   sprintf (time, "%03d:%02d.%03d", (int)min, (int)sec, (int)ms);
   return time;
}
char* format_ts2(INT64 ts, char *time)
{
 UINT32 ms = (UINT32)(ts / 1000000);
 UINT32 sec = ms / 1000;
 UINT32 min = sec / 60;

   sec -= (min*60);
   sprintf (time, "%02d:%02d", (int)min, (int)sec);
   return time;
}
char* format_ts3(u64 ts, char *time)
{
 UINT32 ms = (UINT32)(ts / 1000000);
 UINT32 sec = ms / 1000;
 UINT32 min = sec / 60;
 UINT32 hour = min / 60;
 UINT32 minlim = 60;

   if( min < minlim ) {
      sprintf(time,"%02d:%02d",(int)min,(int)sec-(min*60));
      return time;
   }

   sec -= (min*60);
   min -= (hour*60);
   sprintf (time, "%02d:%02d:%02d", (int)hour, (int)min, (int)sec);
   return time;
}

char* vid_level(UINT32 level)
{
 static char vid[10];
 int i;

   level = (level + 7) / 8;
   for (i=0; i<9; i++) {
      if (level) {
         level--;
         vid[i] = '#';
      }
      else vid[i] = ' ';
   }
   vid[9] = 0;
   return vid;
}

char* aud_level(UINT32 level)
{
 static char aud[50];
 int i;
   memset(aud,0,50);
   level = (level + 199) / 200;
   for (i=0; i<49; i++) {
      if (level) {
         level--;
         aud[i] = '=';
      }
      else aud[i] = ' ';
   }
   aud[49] = 0;
   return aud;
}

// find range to normalize the audio
void normalize(unsigned char *evtbuf, long int evtSize)
{
 EVT_EVENT	event;
 long int size=0;

   evtbuf += 8;			// skip header
   while( size <= evtSize ) {
      memcpy(&event, evtbuf, 24);
      size += 24;
      evtbuf += 24;
      event.type	= SWAP32(event.type);
      event.value1	= SWAP32(event.value1);
      event.value2	= SWAP32(event.value2);
      if (event.type == EVT_AUD) {
         if (event.value1 > audmax) audmax = event.value1;
         if (event.value1 < audmin) audmin = event.value1;
      }
   }

   // audio multiplier
   if (audmax)	factor = 10000 / audmax;
   if (!factor) factor = 1;
   //printf("Scale factor = %d (%d, %d)\n", factor, audmin, audmax);
   audmin = (audmin + ((audmax - audmin)/10)) * factor;	// audio threshold
   //printf("Threshold = %d\n", audmin);
}

// Store evt commercial start & stop times (5K series)
int rtvParseEvt(
   unsigned char *evtbuf, long int evtSize,
   struct evt5kStruct ***s, int *evtcount
)
{
 EVT_EVENT	event;
 UINT32		lastvid = 256;
 UINT32		lastaud = 10000;
 INT64		lasttime, lastvidtime, lastaudtime;
 FADE*		current;
 FADE*		root;
 long int size=0;
 int commercial_start=0, i, j;
 INT64 last_added=0;
 char t1[16], t2[16];
 INT64 addpoint[20];
 int addcount = 0;

   // Normalize audio range
   normalize(evtbuf, evtSize);

   lasttime = lastvidtime = lastaudtime = 0;

   root = current = (FADE*)malloc(sizeof(FADE));
   memset(current, 0, sizeof(FADE));

   // First pass: Create a list of valid fade points within the event groups.
   // We don't care what these fade points are yet.
   evtbuf += 8;			// skip header
   while( size <= evtSize ) {
      memcpy(&event, evtbuf, 24);
      size += 24;
      evtbuf += 24;
      event.timestamp	= SWAP64(event.timestamp);
      event.type	= SWAP32(event.type);
      event.value1	= SWAP32(event.value1);
      event.value2	= SWAP32(event.value2);

      //event.timestamp   = rtv_to_u64(&evtbuf);
      //event.type        = rtv_to_u32(&evtbuf);
      //event.value1      = rtv_to_u32(&evtbuf);
      //event.value2      = rtv_to_u32(&evtbuf);
      //memcpy(&(event.color), evtbuf, 4); evtbuf += 4;
      //size += 24;

      if (!lasttime || (factor == 1 && (event.timestamp - lasttime) > (INT64)1000000000)) {
         // re-use node if not used
         if (current->start) {
            if (!current->stop) current->stop = lasttime;
            current->next = (FADE*)malloc(sizeof(FADE));
            current = current->next;
            memset(current, 0, sizeof(FADE));
         }
         current->stop = 0;
         current->video = 256;
         current->audio = 10000;
         lastvid = 256;
         lastaud = 10000;
      }
      if (event.type == EVT_AUD) {
         if ((event.timestamp - lastvidtime) > 70000000) lastvid = 256;
         lastaud = event.value1 * factor;
         lastaudtime = event.timestamp;
      }
      if (event.type == EVT_VID) {
         int i, divisor = 0;
         if ((event.timestamp - lastaudtime) > 50000000) lastaud = 10000;
         lastvid = 0;
         for (i=0; i<4; i++) {
            if (event.color[i]) {
               lastvid += event.color[i];
               divisor++;
            }
         }
         if (divisor) lastvid /= divisor;
         lastvidtime = event.timestamp;
      }
      if ((factor > 1 && ((!current->start && lastaud < audmin) || (current->start && lastaud >= audmin)))) {
         // re-use node if not used
         if (current->start) {
            if ((event.timestamp - current->start) > 1000000000) {
               if (!current->stop) current->stop = lasttime;
               current->next = (FADE*)malloc(sizeof(FADE));
               current = current->next;
               memset(current, 0, sizeof(FADE));
            }
         }
         current->stop = 0;
         current->video = 256;
         current->audio = 10000;
      }
      // sometimes we see errant past times at the end of the evt,
      // filter them out
      if (event.timestamp >= lasttime) {
         lasttime = event.timestamp;
         // A fade is only valid if BOTH vid and aud fall below
         // our thresholds.
         if (lastvid < 32) {
            if (lastaud < audmin) {
               if (!current->start) {
                  current->start = lasttime;
                  current->video = lastvid;
                  current->audio = lastaud;
               }
               else if (current->stop) {
                  current->start = lasttime;
                  current->video = lastvid;
                  current->audio = lastaud;
                  current->stop = 0;
               }
            }
            else if (current->start && !current->stop) {
               current->stop = lasttime;
            }
         }
         else if ((lastvid > 200) && current->start && !current->stop) {
            current->stop = lasttime;
         }
      }
   }

   // Second pass: Walk the list of fade points and see if anything falls out.
   // For now, just look for time between fades, if it's > 5 minutes it's likely
   // to be a program segment, else it's likely to be a commercial segment.  if
   // there is a fade in the first minute, the recording may have started on a
   // commercial.  nothing fancy.

   current = root;
   lasttime = 0;
   (*evtcount) = 0;

   while (current) {
      if (current->start) {
         INT64 spottime;
         // use the middle value of the fade range as the edit time
         current->stop = (current->stop + current->start)/2;
         spottime = current->stop - lasttime;
         if (!lasttime || spottime >= gMinTime) {
            // if it's less than gProgTime, we want to add the edit
            if (lasttime && spottime < gProgTime) {
               current->type = 1;
            }
            // if it's the first one in the first minute, we want to add
            // the edit
            if (!lasttime && spottime < (INT64)60000000000LL) {
               // 1 min
               current->type = 1;
            }
            else lasttime = current->stop;

            if(RTV_DEBUG) {
               fprintf(stderr,"%c ", current->type ? 'A' : 'D');
               fprintf(stderr,"%s\n", format_ts(current->stop,t1));
            }

            if( current->type ) {
               last_added = current->stop;
               if(last_added) {
                  if(addcount > 20) {
                     addpoint[0] = addpoint[19];
                     addpoint[1] = addpoint[20];
                     addcount = 2;
                  }
                  addpoint[addcount] = last_added;
                  addcount++;
               }
            } else {
               commercial_start = 1;
            }

            if( commercial_start ) {
               if( (*evtcount) == 0 ) {
                  (*s) = (struct evt5kStruct **)malloc(
                     sizeof(struct evt5kStruct *)
                  );
               } else {
                  (*s) = (struct evt5kStruct **)realloc(
                     (*s), sizeof(struct evt5kStruct *)*((*evtcount)+1)
                  );
               }
               (*s)[(*evtcount)] = (struct evt5kStruct *)malloc(
                  sizeof(struct evt5kStruct)
               );
               (*s)[(*evtcount)]->start = current->stop;
               if( last_added && (*evtcount) > 0 ) {
                  (*s)[(*evtcount)-1]->stop = last_added;
               }
               (*evtcount)++;
               commercial_start = 0;
            }
         }
         else lasttime = current->stop;
      }
      current = current->next;
   }

   if( (*evtcount) > 0 ) {
      (*s)[(*evtcount)-1]->stop = lasttime;
      if( last_added > (*s)[(*evtcount)-1]->start )
         (*s)[(*evtcount)-1]->stop = last_added;
   }
   // Throw out last evt point if it's bogus (start time < previous stop time)
   if( (*evtcount) > 1 ) {
      if( (*s)[(*evtcount)-1]->start <= (*s)[(*evtcount)-2]->stop ) {
         (*evtcount)--;
      }
   }
   // last add points > 1 min apart most likely means that last
   // commercial block really ends at the previous add point
   if( addcount > 1 ) {
      addcount--;
      if( addpoint[addcount] - addpoint[addcount-1] > (INT64)60000000000LL ) {
         (*s)[(*evtcount)-1]->stop = addpoint[addcount-1];
      }
   }

   // Sanity check commercial blocks (>6 min diff => throw out)
   for(i=0; i<(*evtcount); ++i) {
      if( (*s)[i]->stop - (*s)[i]->start > (INT64)360000000000LL ) {
         fprintf(stderr,"**Throwing out commercial block: ");
         fprintf(stderr,"%s -> ", format_ts((*s)[i]->start,t1));
         fprintf(stderr,"%s\n", format_ts((*s)[i]->stop,t2));
         for(j=i+1; j<(*evtcount); ++j) {
            (*s)[j-1]->start = (*s)[j]->start;
            (*s)[j-1]->stop = (*s)[j]->stop;
         }
         (*evtcount)--;
      }
   }

   if(RTV_DEBUG) {
      fprintf(stderr,"**5K Commercial EVT points**\n");
      fprintf(stderr,"evtcount = %d\n",(*evtcount));
      for(i=0; i<(*evtcount); ++i) {
         fprintf(stderr,"%d ",i); 
         fprintf(stderr,"(%4.2f mins) ",
            (float)((*s)[i]->stop-(*s)[i]->start)/60e9
         );
         fprintf(stderr,"%s -> ", format_ts((*s)[i]->start,t1));
         fprintf(stderr,"%s\n", format_ts((*s)[i]->stop,t2));
      }
   }

   // release the fade list
   current = root;
   while (current) {
      root = current;
      current = current->next;
      free(root);
   }
   return 1;
}

// NDX FUNCTIONS
// Store ndx timestamp & file position relationships (5K series)
int rtvParse5kNdx(
   unsigned char *ndxbuf, long int ndxSize,
   struct ndx5kStruct ***s, u32 *gopcount
)
{
    int size;
    struct 
    {
        struct ndx5k_record r;
        float  seconds;
    } this;
    u32 recno    = 0;
    u64 basetime = 0;
    
    memset(&this, 0, sizeof this);
    size = read_ndx5k_header(&ndxbuf);

    if(RTV_DEBUG) {
       fprintf(stderr,"%6s %-12s %-10s %6s\n",
          "GOP", "Timestamp", "FilePos", "Iframesize"
       );
       fprintf(stderr,"%6s %-12s %-10s %6s\n",
          "      ", "(mins:secs)", "(bytes)", "(bytes)"
       );
    }

    memset(&this, 0, sizeof this);
    while ( (size += read_ndx5k_record(&ndxbuf, &this.r)) <= ndxSize ) {
        if (recno == 0) {
            basetime   = this.r.timestamp;
            (*s) = (struct ndx5kStruct **)malloc(
               sizeof(struct ndx5kStruct *)
            );
        }
        else {
            (*s) = (struct ndx5kStruct **)realloc(
               (*s), sizeof(struct ndx5kStruct *)*(recno+1)
            );
        }
        this.seconds = ((s64)(this.r.timestamp - basetime)) / 1000000000.0;

        (*s)[recno] = (struct ndx5kStruct *)malloc(sizeof(struct ndx5kStruct));
        (*s)[recno]->time = this.r.timestamp - basetime;
        (*s)[recno]->pos  = this.r.filepos_iframe;

        if(RTV_DEBUG) {
           fprintf(stderr, "%6ld ", recno);
           fprintf(stderr, "%-12s ", format_seconds(this.seconds));
           fprintf(stderr, "%10llu ", this.r.filepos_iframe);
           fprintf(stderr, "%6lu\n", this.r.iframe_size);
        }
        recno++;
        memset(&this, 0, sizeof this);
    }
    *gopcount = recno;
    return 1;
}

// Check if given file name exists
int fileIsFile( char *string ) {
 struct stat status;
   if( stat(string, &status) == -1 ) return 0;
   return(S_ISREG(status.st_mode) && !S_ISDIR(status.st_mode));
}

// Return size of a file in bytes.
long int fileSize( char file[] )
{
struct stat statb;

        if( stat(file, &statb) == -1 )
           return( 0 );
        else
           return( (long int)statb.st_size );
}

// Search ndx for file position corresponding to given time
u64 rtvFindFilePos(
   u64 time, struct ndx5kStruct **ndx,
   u32 *ndxStart, u32 ndxcount
)
{
 int i;
 u64 d1, d2;
 u64 tlast = -1;
 u64 pos = -1;
   for(i=(*ndxStart); i<ndxcount; ++i) {
      if( ndx[i]->time >= time ) {
         d1 = ndx[i]->time - time;
         d2 = time - tlast;
         if( i > 0 && d1 > d2 ) {
            (*ndxStart) = i-1;
            return ndx[i-1]->pos;
         } else {
            (*ndxStart) = i;
            return ndx[i]->pos;
         }
      }
      tlast = ndx[i]->time;
   }
   return pos;
}

// Obtain commercial mark start & stop points (time and file position)
// Return 1 for normal exit, 0 otherwise.
int rtvGetCommercialMarks(
   input_thread_t *p_input,
   unsigned char *evtbuf, long int evtSize,
   unsigned char *ndxbuf, long int ndxSize,
   struct rtvCommercialMarks ***s, int *count
)
{
 struct evt5kStruct **evt5k;
 struct ndx5kStruct **ndx5k;
 u32 ndxStart=0;
 u32 ndxcount;
 int evtcount, i;
 u64 fileStart, fileStop;
 char t1[16];
 u64 minCommercial=20000000000LLU;

   // Nothing to do if both buffers empty
   if(evtSize <= 0 && ndxSize <= 0) return 0;

   if(evtSize > 0) {
      // 5K series processing

      // Mark as 5K series
      p_input->p_libvlc->rtvModel = 5;

      // Store evt file contents in memory
      if( !rtvParseEvt(evtbuf, evtSize, &evt5k, &evtcount) ) return 0;

      if( evtcount > 0 ) {
         // Store ndx file contents in memory
         if( !rtvParse5kNdx(ndxbuf, ndxSize, &ndx5k, &ndxcount) ) return 0;

         // Build array of commercial start stop times & file positions
         (*s) = (struct rtvCommercialMarks **)malloc(
            sizeof(struct rtvCommercialMarks *)*(evtcount)
         );
         (*count) = 0;
         if(RTV_DEBUG) fprintf(stderr,"** 5K Commercial Marks**\n");
         for( i=0; i<evtcount; i++ ) {
            if( evt5k[i]->stop > evt5k[i]->start && (evt5k[i]->stop - evt5k[i]->start) >= minCommercial ) {
               fileStart = rtvFindFilePos(
                  evt5k[i]->start, ndx5k, &ndxStart, ndxcount
               );
               fileStop  = rtvFindFilePos(
                  evt5k[i]->stop, ndx5k, &ndxStart, ndxcount
               );
               (*s)[(*count)] = (struct rtvCommercialMarks *)malloc(
                  sizeof(struct rtvCommercialMarks)
               );
               (*s)[(*count)]->tstart = evt5k[i]->start;
               (*s)[(*count)]->fstart = fileStart;
               (*s)[(*count)]->tstop  = evt5k[i]->stop;
               (*s)[(*count)]->fstop  = fileStop;

               if(RTV_DEBUG) {
                  fprintf(stderr,"%d: ",*count);
                  fprintf(stderr,"tstart=%s ",
                     format_ts((*s)[*count]->tstart,t1)
                  );
                  fprintf(stderr,"fstart=%llu ", (*s)[*count]->fstart);
		  fprintf(stderr,"tstop=%s ",
		     format_ts((*s)[*count]->tstop,t1)
		  );
                  fprintf(stderr,"fstop=%llu\n", (*s)[*count]->fstop);
               }
               (*count)++;
            }
         }

         // Free up allocated memory
         if(evtcount > 0) {
            for(i=0; i<evtcount; ++i)
               free(evt5k[i]);
            free(evt5k);
         }
         if(ndxcount > 0) {
            for(i=0; i<ndxcount; ++i)
               free(ndx5k[i]);
            free(ndx5k);
         }
      }
      else {
         if(RTV_DEBUG) fprintf(stderr,"No commercial points found!\n");
      }
   }

   else {
      // 4K series processing

      // Mark as 4K series
      p_input->p_libvlc->rtvModel = 4;

      if( !rtvParse4kNdx(ndxbuf, ndxSize, s, count, minCommercial) ) return 0;
      if(RTV_DEBUG) {
         fprintf(stderr,"** 4K Commercial Marks**\n");
         for( i=0; i<(*count); i++ ) {
            fprintf(stderr,"%d: ",i);
            fprintf(stderr,"tstart=%s ",
               format_ts((*s)[i]->tstart,t1)
            );
            fprintf(stderr,"fstart=%llu ", (*s)[i]->fstart);
            fprintf(stderr,"tstop=%s ",
               format_ts((*s)[i]->tstop,t1)
            );
            fprintf(stderr,"fstop=%llu\n", (*s)[i]->fstop);
         }
      }
   }

   return 1;
}

// Identify if given time is within commercial bounds or not
// If within commercial bounds, return it's index in commercial array
// else return -1
int rtvIsInCommercial(u64 time, struct rtvCommercialMarks **s, int count)
{
 int i;
   for(i=0; i<count; ++i) {
      if( time >= s[i]->tstart && time <= s[i]->tstop ) {
         return i;
      }
   }
   return -1;
}

// Save entire file contents to unsigned char buffer
int rtvFileToBuf(char *file, unsigned char **buf, long int *size)
{
 FILE *ifp;
 char msg[256];

   if( !fileIsFile(file) ) return(0);
   (*size) = fileSize(file);
   ifp = fopen(file, "rb");
   (*buf) = (unsigned char *)malloc((*size)+1);
   if( !fread((*buf), (*size), 1, ifp) || (*size) == 0 ) {
      sprintf(msg, "Failed to read file: %s",file);
      return(rtvError(msg));
   }
   if(ifp) fclose(ifp);
   return(1);
}

// Add bookmarks at all commercial stop points
void rtvAddCABookmarks(input_thread_t *p_input)
{
   if(p_input->p_libvlc->rtvmarks_num > 0) {
      seekpoint_t seekpoint;
      int i;
      char t1[16], t2[16], t3[16], t4[16];

      seekpoint.psz_name = (char *)malloc(50*sizeof(char));

      // Add time 0 bookmark
      strcpy(seekpoint.psz_name,"Program Start");
      seekpoint.i_byte_offset = 0;
      seekpoint.i_time_offset = 0;
      input_Control( p_input, INPUT_ADD_BOOKMARK, &seekpoint );

      for(i=0; i<p_input->p_libvlc->rtvmarks_num; ++i) {
         sprintf(t1,"%s-",
            format_ts3(p_input->p_libvlc->rtvmarks[i]->tstart,t4)
         );
         sprintf(t2,"%s ",
            format_ts3(p_input->p_libvlc->rtvmarks[i]->tstop,t4)
         );
         sprintf(t3,"(%s)",
            format_ts2(p_input->p_libvlc->rtvmarks[i]->tstop - p_input->p_libvlc->rtvmarks[i]->tstart,t4)
         );
         sprintf(seekpoint.psz_name,"CA%d %s%s%s",i+1,t1,t2,t3);
         seekpoint.i_byte_offset = p_input->p_libvlc->rtvmarks[i]->fstop;
         seekpoint.i_time_offset = p_input->p_libvlc->rtvmarks[i]->tstop/1000;
         input_Control( p_input, INPUT_ADD_BOOKMARK, &seekpoint );
      }

      free(seekpoint.psz_name);
   }
}

// Figure out RTV commercial points from .evt/.ndx file contents
int rtvSetupCommercialPointsFromFiles(input_thread_t *p_input, char *mpg) {
 char *base, *evtFile, *ndxFile;
 unsigned char *evtbuf, *ndxbuf;
 long int evtSize=0, ndxSize=0;
 int i;

   // Strip off .mpg
   fprintf(stderr,"%s\n",mpg);
   base = (char *)malloc(strlen(mpg)+1);
   evtFile = (char *)malloc(strlen(mpg)+1);
   ndxFile = (char *)malloc(strlen(mpg)+1);
   strncpy(base, mpg, strlen(mpg)-4);
   base[strlen(mpg)-4] = '\0';
   sprintf(evtFile,"%s.evt",base);
   sprintf(ndxFile,"%s.ndx",base);

   RTV_DEBUG = 1;

   rtvFileToBuf(evtFile, &evtbuf, &evtSize);
   if( !rtvFileToBuf(ndxFile, &ndxbuf, &ndxSize) ) return 0;

   i = rtvGetCommercialMarks(
      p_input,
      evtbuf, evtSize, ndxbuf, ndxSize,
      &p_input->p_libvlc->rtvmarks, &p_input->p_libvlc->rtvmarks_num
   );
   if(!i) return 0;

   // Free up allocated memory
   free(evtFile); free(ndxFile); free(base);
   if(evtSize>0) free(evtbuf);
   if(ndxSize>0) free(ndxbuf);
   return 1;
}

// Figure out RTV commercial points from .evt/.ndx
// contents obtained via DVA (in modules/access/http.c)
int rtvSetupCommercialPointsFromDVA(input_thread_t *p_input) {
 int i;

   RTV_DEBUG = 1;
   i = rtvGetCommercialMarks(
      p_input,
      p_input->p_libvlc->evtBuf, p_input->p_libvlc->evtSize,
      p_input->p_libvlc->ndxBuf, p_input->p_libvlc->ndxSize,
      &p_input->p_libvlc->rtvmarks, &p_input->p_libvlc->rtvmarks_num
   );

   // Free up allocated memory
   if(p_input->p_libvlc->evtSize>0) free(p_input->p_libvlc->evtBuf);
   if(p_input->p_libvlc->ndxSize>0) free(p_input->p_libvlc->ndxBuf);
   if(!i) return 0;
   return 1;
}

// Jump ahead to next closest commercial block end
void rtvJumpToNextBookmark( input_thread_t *p_input )
{
   if(p_input->p_libvlc->rtvmarks_num > 0 && p_input) {
      int i;
      int64_t pos = stream_Tell(p_input->input.p_demux->s);
      for(i=0; i<p_input->p_libvlc->rtvmarks_num; ++i) {
         if(p_input->p_libvlc->rtvmarks[i]->fstop > pos) {
            char msg[256], t1[16];
            stream_Seek(p_input->input.p_demux->s,p_input->p_libvlc->rtvmarks[i]->fstop);
            sprintf(msg,">> %s",format_ts3(p_input->p_libvlc->rtvmarks[i]->tstop,t1));
            vout_OSDMessage( p_input->input.p_demux->s, DEFAULT_CHAN, _( msg ) );
            return;
         }
      }
   }
}

// Jump backwards to previous closest commercial block end
void rtvJumpToPrevBookmark( input_thread_t *p_input )
{
   if(p_input->p_libvlc->rtvmarks_num > 0 && p_input) {
      int i;
      int64_t pos = stream_Tell(p_input->input.p_demux->s);
      int64_t margin = 3000000; // Provide some margin for multiple back skips
      char msg[256], t1[16];

      // Jump back to time 0 if pos <= 1st CA bookmark + margin
      if( pos < p_input->p_libvlc->rtvmarks[0]->fstop + margin ) {
         stream_Seek(p_input->input.p_demux->s,0);
         sprintf(msg,"<< %s",format_ts3(0,t1));
         vout_OSDMessage( p_input->input.p_demux->s, DEFAULT_CHAN, _( msg ) );
         return;
      }

      // Else jump back to previous CA bookmark
      for(i=p_input->p_libvlc->rtvmarks_num-1; i>=0; --i) {
         if(p_input->p_libvlc->rtvmarks[i]->fstop < pos) {
            if( i > 0 && pos < p_input->p_libvlc->rtvmarks[i]->fstop + margin ) {
               stream_Seek(p_input->input.p_demux->s,p_input->p_libvlc->rtvmarks[i-1]->fstop);
               sprintf(msg,"<< %s",format_ts3(p_input->p_libvlc->rtvmarks[i-1]->tstop,t1));
               vout_OSDMessage( p_input->input.p_demux->s, DEFAULT_CHAN, _( msg ) );
            }
            else {
               stream_Seek(p_input->input.p_demux->s,p_input->p_libvlc->rtvmarks[i]->fstop);
               sprintf(msg,"<< %s",format_ts3(p_input->p_libvlc->rtvmarks[i]->tstop,t1));
               vout_OSDMessage( p_input->input.p_demux->s, DEFAULT_CHAN, _( msg ) );
            }
            return;
         }
      }
   }
}

// If requested (via hotkey) jump to next/prev CA bookmark
void rtvHandleBookmarkJump( input_thread_t *p_input )
{
   if(p_input->p_libvlc->jumpToNextCA == 1) {
      rtvJumpToNextBookmark(p_input);
      p_input->p_libvlc->jumpToNextCA = 0;
   }

   if(p_input->p_libvlc->jumpToPrevCA == 1) {
      rtvJumpToPrevBookmark(p_input);
      p_input->p_libvlc->jumpToPrevCA = 0;
   }
}

void rtvHandleCA( input_thread_t *p_input )
{
 char t1[16];
 if( p_input->p_libvlc->disableCA != 1 ) {
   if(p_input->p_libvlc->rtvmarks_num > 0 && p_input) {
      int i;
      int64_t pos;
      pos = stream_Tell(p_input->input.p_demux->s);
      for(i=0; i<p_input->p_libvlc->rtvmarks_num; ++i) {
         if(pos >= p_input->p_libvlc->rtvmarks[i]->fstart && pos <= p_input->p_libvlc->rtvmarks[i]->fstop) {
            char msg[256];
            int64_t dt = p_input->p_libvlc->rtvmarks[i]->tstop-p_input->p_libvlc->rtvmarks[i]->tstart;
            stream_Seek(p_input->input.p_demux->s,p_input->p_libvlc->rtvmarks[i]->fstop);
            sprintf(msg,">>> %s",format_ts2(dt,t1));
            vout_OSDMessage( p_input->input.p_demux->s, DEFAULT_CHAN, _( msg ) );
            fprintf(stderr,"%s\n",msg);
         }
      }
   }
 }
}
/* moyekj END */

/*****************************************************************************
 * Open
 *****************************************************************************/
static int Open( vlc_object_t *p_this )
{
    demux_t     *p_demux = (demux_t*)p_this;
    demux_sys_t *p_sys;
    /* moyekj START */
    input_thread_t *p_input = vlc_object_find(
       p_demux, VLC_OBJECT_INPUT, FIND_PARENT
    );
    /* moyekj END */

    uint8_t     *p_peek;

    if( stream_Peek( p_demux->s, &p_peek, 4 ) < 4 )
    {
        msg_Err( p_demux, "cannot peek" );
        return VLC_EGENERIC;
    }

    if( p_peek[0] != 0 || p_peek[1] != 0 ||
        p_peek[2] != 1 || p_peek[3] < 0xb9 )
    {
        msg_Warn( p_demux, "this does not look like an MPEG PS stream, "
                  "continuing anyway" );
    }

    /* Fill p_demux field */
    p_demux->pf_demux = Demux;
    p_demux->pf_control = Control;
    p_demux->p_sys = p_sys = malloc( sizeof( demux_sys_t ) );
    /* moyekj start */
    p_input->p_libvlc->rtv_base_time = -1;
    p_input->p_libvlc->rtv_base_last = -1;
    p_input->p_libvlc->rtv_base_offset = -1;
    p_input->p_libvlc->rtv_base_count = 0;
    p_input->p_libvlc->rtvmarks_num = -1;
    /* moyekj end */

    /* Init p_sys */
    p_sys->i_mux_rate = 0;
    p_sys->i_scr      = -1;
    p_sys->b_lost_sync = VLC_FALSE;
    p_sys->b_have_pack = VLC_FALSE;

    ps_psm_init( &p_sys->psm );
    ps_track_init( p_sys->tk );

    /* TODO prescanning of ES */

    return VLC_SUCCESS;
}

static int OpenAlt( vlc_object_t *p_this )
{
    demux_t *p_demux = (demux_t*)p_this;
    uint8_t *p_peek;

    if( stream_Peek( p_demux->s, &p_peek, 4 ) < 4 )
    {
        msg_Err( p_demux, "cannot peek" );
        return VLC_EGENERIC;
    }

    if( p_peek[0] != 0 || p_peek[1] != 0 ||
        p_peek[2] != 1 || p_peek[3] < 0xb9 )
    {
        if( !p_demux->b_force ) return VLC_EGENERIC;
    }

    return Open( p_this );
}

/*****************************************************************************
 * Close
 *****************************************************************************/
static void Close( vlc_object_t *p_this )
{
    demux_t     *p_demux = (demux_t*)p_this;
    demux_sys_t *p_sys = p_demux->p_sys;
    int i;

    for( i = 0; i < PS_TK_COUNT; i++ )
    {
        ps_track_t *tk = &p_sys->tk[i];
        if( tk->b_seen )
        {
            es_format_Clean( &tk->fmt );
            if( tk->es ) es_out_Del( p_demux->out, tk->es );
        }
    }

    ps_psm_destroy( &p_sys->psm );

    free( p_sys );
}

/*****************************************************************************
 * Demux:
 *****************************************************************************/
static int Demux( demux_t *p_demux )
{
    demux_sys_t *p_sys = p_demux->p_sys;
    int i_ret, i_id, i_mux_rate;
    uint32_t i_code;
    block_t *p_pkt;
    /* moyekj START */
    input_thread_t *p_input = vlc_object_find(
       p_demux, VLC_OBJECT_INPUT, FIND_PARENT
    );
    /* moyekj END */

    i_ret = ps_pkt_resynch( p_demux->s, &i_code );
    if( i_ret < 0 )
    {
        return 0;
    }
    else if( i_ret == 0 )
    {
        if( !p_sys->b_lost_sync )
            msg_Warn( p_demux, "garbage at input, trying to resync..." );

        p_sys->b_lost_sync = VLC_TRUE;
        return 1;
    }

    if( p_sys->b_lost_sync ) msg_Warn( p_demux, "found sync code" );
    p_sys->b_lost_sync = VLC_FALSE;

    if( ( p_pkt = ps_pkt_read( p_demux->s, i_code ) ) == NULL )
    {
        return 0;
    }

    switch( i_code )
    {
    case 0x1b9:
        block_Release( p_pkt );
        break;

    case 0x1ba:
        if( !ps_pkt_parse_pack( p_pkt, &p_sys->i_scr, &i_mux_rate ) )
        {
            if( !p_sys->b_have_pack ) p_sys->b_have_pack = VLC_TRUE;
            /* done later on to work around bad vcd/svcd streams */
            /* es_out_Control( p_demux->out, ES_OUT_SET_PCR, p_sys->i_scr ); */

            /* moyekj start */

            // If ReplayTV mpeg stream, setup CA points & bookmarks
            if(p_input->p_libvlc->rtvmarks_num == -1) {
               p_input->p_libvlc->rtvmarks_num = 0;
               p_input->p_libvlc->rtvModel = 0;
               if( ! p_input->p_libvlc->ndxSize ) {
                  // Figure out RTV commercial points from .ndx/.evt files
                  rtvSetupCommercialPointsFromFiles(
                     p_input, p_demux->psz_path
                  );
               }
               else {
                  // Figure out RTV commercial points from .ndx/.evt
                  // contents obtained from DVA (modules/access/http.c)
                  rtvSetupCommercialPointsFromDVA(p_input);
               }
               rtvAddCABookmarks(p_input);
            }
            if(p_input->p_libvlc->rtvmarks_num > 0) {
               rtvHandleCA(p_input);
               rtvHandleBookmarkJump(p_input);
            }

            // Prevent elapsed/total times from jumping all over the place
            // by limiting mux rate changes
            if( p_input->p_libvlc->rtv_base_time == -1 ) {
               p_input->p_libvlc->rtv_base_time = p_sys->i_scr;
               p_input->p_libvlc->rtv_base_offset = stream_Tell(p_demux->s);
               if( p_input->p_libvlc->rtv_base_time > 0 ) {
                  p_input->p_libvlc->rtv_base_last =
                     20000*p_input->p_libvlc->rtv_base_offset/p_input->p_libvlc->rtv_base_time;
               }
            }
            if( p_sys->i_scr != p_input->p_libvlc->rtv_base_time &&
                p_input->p_libvlc->rtv_base_count > 50 ) {
               int64_t fdelta =
                  stream_Tell(p_demux->s) - p_input->p_libvlc->rtv_base_offset;
               fdelta *= 20000;
               int64_t tdelta = p_sys->i_scr - p_input->p_libvlc->rtv_base_time;
               if( fdelta > 0 && tdelta > 0 )
                  i_mux_rate = (int)(fdelta / tdelta);
               p_input->p_libvlc->rtv_base_count = 0;
               p_input->p_libvlc->rtv_base_last = i_mux_rate;
            } else {
               i_mux_rate = p_input->p_libvlc->rtv_base_last;
            }
            p_input->p_libvlc->rtv_base_count++;
            /* moyekj end */

            if( i_mux_rate > 0 ) p_sys->i_mux_rate = i_mux_rate;
        }
        block_Release( p_pkt );
        break;

    case 0x1bb:
        if( !ps_pkt_parse_system( p_pkt, &p_sys->psm, p_sys->tk ) )
        {
            int i;
            for( i = 0; i < PS_TK_COUNT; i++ )
            {
                ps_track_t *tk = &p_sys->tk[i];

                if( tk->b_seen && !tk->es && tk->fmt.i_cat != UNKNOWN_ES )
                {
                    tk->es = es_out_Add( p_demux->out, &tk->fmt );
                }
            }
        }
        block_Release( p_pkt );
        break;

    case 0x1bc:
        if( p_sys->psm.i_version == 0xFFFF )
            msg_Dbg( p_demux, "contains a PSM");

        ps_psm_fill( &p_sys->psm, p_pkt, p_sys->tk, p_demux->out );
        block_Release( p_pkt );
        break;

    default:
        if( (i_id = ps_pkt_id( p_pkt )) >= 0xc0 )
        {
            vlc_bool_t b_new = VLC_FALSE;
            ps_track_t *tk = &p_sys->tk[PS_ID_TO_TK(i_id)];

            if( !tk->b_seen )
            {
                if( !ps_track_fill( tk, &p_sys->psm, i_id ) )
                {
                    tk->es = es_out_Add( p_demux->out, &tk->fmt );
                    b_new = VLC_TRUE;
                }
                else
                {
                    msg_Dbg( p_demux, "es id=0x%x format unknown", i_id );
                }
                tk->b_seen = VLC_TRUE;
            }

            /* The popular VCD/SVCD subtitling WinSubMux does not
             * renumber the SCRs when merging subtitles into the PES */
            if( tk->b_seen &&
                ( tk->fmt.i_codec == VLC_FOURCC('o','g','t',' ') ||
                  tk->fmt.i_codec == VLC_FOURCC('c','v','d',' ') ) )
            {
                p_sys->i_scr = -1;
            }

            if( p_sys->i_scr > 0 )
                es_out_Control( p_demux->out, ES_OUT_SET_PCR, p_sys->i_scr );

            p_sys->i_scr = -1;

            if( tk->b_seen && tk->es &&
                !ps_pkt_parse_pes( p_pkt, tk->i_skip ) )
            {
                if( !b_new && !p_sys->b_have_pack && tk->fmt.i_cat == AUDIO_ES && p_pkt->i_pts > 0 )
                {
                    /* A hack to sync the A/V on PES files. */
                    msg_Dbg( p_demux, "force SCR: %lld", p_pkt->i_pts );
                    es_out_Control( p_demux->out, ES_OUT_SET_PCR, p_pkt->i_pts );
                }

                es_out_Send( p_demux->out, tk->es, p_pkt );
            }
            else
            {
                block_Release( p_pkt );
            }
        }
        else
        {
            block_Release( p_pkt );
        }
        break;
    }

    return 1;
}

/*****************************************************************************
 * Control:
 *****************************************************************************/
static int Control( demux_t *p_demux, int i_query, va_list args )
{
    demux_sys_t *p_sys = p_demux->p_sys;
    double f, *pf;
    int64_t i64, *pi64;

    switch( i_query )
    {
        case DEMUX_GET_POSITION:
            pf = (double*) va_arg( args, double* );
            i64 = stream_Size( p_demux->s );
            if( i64 > 0 )
            {
                *pf = (double)stream_Tell( p_demux->s ) / (double)i64;
            }
            else
            {
                *pf = 0.0;
            }
            return VLC_SUCCESS;

        case DEMUX_SET_POSITION:
            f = (double) va_arg( args, double );
            i64 = stream_Size( p_demux->s );

            es_out_Control( p_demux->out, ES_OUT_RESET_PCR );

            return stream_Seek( p_demux->s, (int64_t)(i64 * f) );

        case DEMUX_GET_TIME:
            pi64 = (int64_t*)va_arg( args, int64_t * );
            if( p_sys->i_mux_rate > 0 )
            {
                *pi64 = (int64_t)1000000 * ( stream_Tell( p_demux->s ) / 50 ) /
                    p_sys->i_mux_rate;
                return VLC_SUCCESS;
            }
            *pi64 = 0;
            return VLC_EGENERIC;

        case DEMUX_GET_LENGTH:
            pi64 = (int64_t*)va_arg( args, int64_t * );
            if( p_sys->i_mux_rate > 0 )
            {
                *pi64 = (int64_t)1000000 * ( stream_Size( p_demux->s ) / 50 ) /
                    p_sys->i_mux_rate;
                return VLC_SUCCESS;
            }
            *pi64 = 0;
            return VLC_EGENERIC;

        case DEMUX_SET_TIME:
        case DEMUX_GET_FPS:
        default:
            return VLC_EGENERIC;
    }
}

/*****************************************************************************
 * Divers:
 *****************************************************************************/

/* PSResynch: resynch on a system starcode
 *  It doesn't skip more than 512 bytes
 *  -1 -> error, 0 -> not synch, 1 -> ok
 */
static int ps_pkt_resynch( stream_t *s, uint32_t *pi_code )
{
    uint8_t *p_peek;
    int     i_peek;
    int     i_skip;

    if( stream_Peek( s, &p_peek, 4 ) < 4 )
    {
        return -1;
    }
    if( p_peek[0] == 0 && p_peek[1] == 0 && p_peek[2] == 1 &&
        p_peek[3] >= 0xb9 )
    {
        *pi_code = 0x100 | p_peek[3];
        return 1;
    }

    if( ( i_peek = stream_Peek( s, &p_peek, 512 ) ) < 4 )
    {
        return -1;
    }
    i_skip = 0;

    for( ;; )
    {
        if( i_peek < 4 )
        {
            break;
        }
        if( p_peek[0] == 0 && p_peek[1] == 0 && p_peek[2] == 1 &&
            p_peek[3] >= 0xb9 )
        {
            *pi_code = 0x100 | p_peek[3];
            return stream_Read( s, NULL, i_skip ) == i_skip ? 1 : -1;
        }

        p_peek++;
        i_peek--;
        i_skip++;
    }
    return stream_Read( s, NULL, i_skip ) == i_skip ? 0 : -1;
}

static block_t *ps_pkt_read( stream_t *s, uint32_t i_code )
{
    uint8_t *p_peek;
    int      i_peek = stream_Peek( s, &p_peek, 14 );
    int      i_size = ps_pkt_size( p_peek, i_peek );

    if( i_size <= 6 && p_peek[3] > 0xba )
    {
        /* Special case, search the next start code */
        i_size = 6;
        for( ;; )
        {
            i_peek = stream_Peek( s, &p_peek, i_size + 1024 );
            if( i_peek <= i_size + 4 )
            {
                return NULL;
            }
            while( i_size <= i_peek - 4 )
            {
                if( p_peek[i_size] == 0x00 && p_peek[i_size+1] == 0x00 &&
                    p_peek[i_size+2] == 0x01 && p_peek[i_size+3] >= 0xb9 )
                {
                    return stream_Block( s, i_size );
                }
                i_size++;
            }
        }
    }
    else
    {
        /* Normal case */
        return stream_Block( s, i_size );
    }

    return NULL;
}
